#!/usr/bin/env python3
"""Klokku macOS Status Bar App - switch budget items from your menu bar."""

import asyncio
import logging
import os
from pathlib import Path

import rumps
from klokku_python_client import KlokkuApi, WeeklyItem, CurrentEvent

logging.basicConfig(level=logging.WARNING)
logger = logging.getLogger("klokku-macos")

CONFIG_DIR = os.path.expanduser("~/.config/klokku-macos")
CONFIG_FILE = os.path.join(CONFIG_DIR, "config")
DEFAULT_URL = "https://app.klokku.com"
REFRESH_INTERVAL = 60  # seconds


def _statusbar_icon_path() -> str | None:
    """Return icon path for both dev runs and py2app bundle runs."""
    # In a py2app bundle, RESOURCEPATH points to: *.app/Contents/Resources
    resource_path = os.environ.get("RESOURCEPATH")
    if resource_path:
        bundled = Path(resource_path) / "statusbar_icon.png"
        if bundled.exists():
            return str(bundled)

    # When running from source
    dev = Path(__file__).resolve().parent / "resources" / "statusbar_icon.png"
    if dev.exists():
        return str(dev)

    return None


def load_config() -> dict:
    """Load configuration from environment variables or config file."""
    config = {
        "url": os.environ.get("KLOKKU_URL", ""),
        "token": os.environ.get("KLOKKU_TOKEN", ""),
    }

    if not config["token"] and os.path.exists(CONFIG_FILE):
        with open(CONFIG_FILE) as f:
            for line in f:
                line = line.strip()
                if not line or line.startswith("#"):
                    continue
                if "=" in line:
                    key, value = line.split("=", 1)
                    config[key.strip()] = value.strip()

    if not config["url"]:
        config["url"] = DEFAULT_URL

    return config


def save_config(url: str, token: str):
    """Save configuration to the config file."""
    os.makedirs(CONFIG_DIR, exist_ok=True)
    with open(CONFIG_FILE, "w") as f:
        f.write(f"url={url}\n")
        f.write(f"token={token}\n")


class KlokkuStatusBarApp(rumps.App):
    def __init__(self):
        super().__init__(
            "Klokku",
            title="",
            quit_button=None,
            icon=_statusbar_icon_path(),
            template=True,
        )

        config = load_config()
        self.url = config["url"]
        self.token = config["token"]
        self.weekly_items: list[WeeklyItem] = []
        self.current_event: CurrentEvent | None = None
        self._configured = False

        if self.token:
            self._do_refresh()
        else:
            self._rebuild_menu()
            self._open_settings()

        self._timer = rumps.Timer(self._on_timer, REFRESH_INTERVAL)
        self._timer.start()

    def _on_timer(self, _):
        if self._configured:
            self._do_refresh()

    def _do_refresh(self):
        """Fetch latest data from Klokku and rebuild the menu."""
        try:
            asyncio.run(self._fetch_data())
        except Exception as e:
            logger.error(f"Refresh failed: {e}")
            self.title = "⚠"
        self._rebuild_menu()

    async def _fetch_data(self):
        """Fetch the weekly plan and current event from the API."""
        async with KlokkuApi(self.url) as api:
            authenticated = await api.authenticate(self.token)
            if not authenticated:
                logger.error("Authentication failed")
                self.title = "Auth Error"
                self._configured = False
                return

            self._configured = True

            plan = await api.get_current_week_plan()
            if plan:
                self.weekly_items = sorted(plan.items, key=lambda x: x.position)

            event = await api.get_current_event()
            self.current_event = event

            if event:
                self.title = f"{event.planItem.name}"
            else:
                self.title = "--"

    def _rebuild_menu(self):
        """Rebuild the dropdown menu with current budget items."""
        self.menu.clear()

        current_budget_id = None
        if self.current_event:
            current_budget_id = self.current_event.planItem.budgetItemId

        for item in self.weekly_items:
            label = item.name
            menu_item = rumps.MenuItem(
                label,
                callback=self._make_switch_callback(item.budgetItemId),
            )
            if item.budgetItemId == current_budget_id:
                menu_item.state = 1  # checkmark
            self.menu.add(menu_item)

        if self.weekly_items:
            self.menu.add(rumps.separator)

        self.menu.add(rumps.MenuItem("Refresh", callback=self._on_refresh))
        self.menu.add(rumps.MenuItem("Settings\u2026", callback=lambda _: self._open_settings()))
        self.menu.add(rumps.separator)
        self.menu.add(rumps.MenuItem("Quit", callback=self._on_quit))

    def _open_settings(self):
        """Open the settings dialog for URL and token configuration."""
        url_window = rumps.Window(
            message="Enter the Klokku server URL:",
            title="Klokku Settings - Server URL",
            default_text=self.url or DEFAULT_URL,
            ok="Next",
            cancel="Cancel",
            dimensions=(380, 24),
        )
        url_response = url_window.run()
        if not url_response.clicked:
            return

        new_url = url_response.text.strip() or DEFAULT_URL

        token_window = rumps.Window(
            message="Enter your personal access token (starts with pat.):",
            title="Klokku Settings - Access Token",
            default_text=self.token or "",
            ok="Save",
            cancel="Cancel",
            dimensions=(380, 24),
        )
        token_response = token_window.run()
        if not token_response.clicked:
            return

        new_token = token_response.text.strip()
        if not new_token:
            rumps.alert(
                title="Klokku Settings",
                message="Token cannot be empty.",
            )
            return

        self.url = new_url
        self.token = new_token
        save_config(self.url, self.token)

        self.title = "..."
        self._do_refresh()

    def _make_switch_callback(self, budget_item_id: int):
        """Create a callback that switches to the given budget item."""
        def callback(_):
            prev_title = self.title
            self.title = "..."
            try:
                asyncio.run(self._switch_event(budget_item_id))
                self._do_refresh()
            except Exception as e:
                logger.error(f"Failed to switch event: {e}")
                self.title = prev_title
                rumps.notification(
                    "Klokku",
                    "Error",
                    f"Failed to switch budget item: {e}",
                )
        return callback

    async def _switch_event(self, budget_item_id: int):
        """Switch the current event to the given budget item."""
        async with KlokkuApi(self.url) as api:
            await api.authenticate(self.token)
            await api.set_current_event(budget_item_id)

    def _on_refresh(self, _):
        self._do_refresh()

    @staticmethod
    def _on_quit(_):
        rumps.quit_application()


if __name__ == "__main__":
    KlokkuStatusBarApp().run()
